<?php

namespace App\Http\Controllers;

use App\Models\Admission;
use App\Models\Classe;
use App\Models\Section;
use App\Models\Student;
use App\Models\Guardian;
use Illuminate\Http\Request;
use App\Models\Fee_structure;
use App\Models\School_setting;
use App\Models\Student_classe;
use App\Models\Emergency_contact;
use App\Models\Previous_education;
use App\Models\Teacher_timetable;
use Illuminate\Support\Facades\DB;
use App\Models\Student_cards_template;
use App\Models\AdmissionHistory;
use App\Models\School_session;
use App\Models\Student_assign_fee;
use App\Models\Student_fee;
use App\Models\Student_fee_detail;
use App\Models\Transaction;
use App\Models\Transaction_detail;
use Barryvdh\DomPDF\Facade\Pdf;

class StudentController extends Controller
{
    public function currentStudent(Request $request)
    {
        $students = Student::with(['previousEducation', 'guardian', 'student_classes', 'classes', 'sections'])
            ->whereHas('latestAdmission')
            ->with('latestAdmission')
            ->where('status', 'active');
        // dd($students);
        $schoolSettings = School_setting::all()->pluck('setting_value', 'setting_key');
        // Apply search filter
        if ($request->filled('search_query')) {
            $students->where(function ($query) use ($request) {
                $query->where('student_name', 'like', '%' . $request->search_query . '%')
                    ->orWhere('email', 'like', '%' . $request->search_query . '%');
            });
        }

        // Apply class filter if selected
        if ($request->filled('class_filter')) {
            $students->whereHas('latestAdmission', function ($query) use ($request) {
                $query->where('class_id', $request->class_filter);
            });
        }

        // Apply section filter if selected
        if ($request->filled('section_filter')) {
            $students->whereHas('latestAdmission', function ($query) use ($request) {
                $query->where('section_id', $request->section_filter);
            });
        }


        $students = $students->paginate(5);
        //dd($students);

        if ($request->ajax()) {
            return view('student.current-student-search-results', compact('students'))->render();
        }

        $classes  = DB::table('classes')->get();
        $sections = DB::table('sections')->get();

        return view('student.current-students', compact('students', 'classes', 'sections', 'schoolSettings'));
    }

    public function pdfCurrentStudent(Request $request)
    {
        $students = Student::with(['previousEducation', 'guardian', 'student_classes', 'classes', 'sections'])
            ->whereHas('latestAdmission')
            ->with('latestAdmission')
            ->where('status', 'active');

        // Apply search filter
        if ($request->filled('search_query')) {
            $students->where(function ($query) use ($request) {
                $query->where('student_name', 'like', '%' . $request->search_query . '%')
                    ->orWhere('email', 'like', '%' . $request->search_query . '%');
            });
        }

        // Apply class filter if selected
        if ($request->filled('class_filter')) {
            $students->whereHas('latestAdmission', function ($query) use ($request) {
                $query->where('class_id', $request->class_filter);
            });
        }

        // Apply section filter if selected
        if ($request->filled('section_filter')) {
            $students->whereHas('latestAdmission', function ($query) use ($request) {
                $query->where('section_id', $request->section_filter);
            });
        }

        // No pagination for PDF
        $students = $students->get();

        // Fetch settings
        $schoolSettings = School_setting::all()->pluck('setting_value', 'setting_key');

        // Generate PDF
        $pdf = Pdf::loadView('student.pdfCurrentStudent', compact('students', 'schoolSettings'));

        return $pdf->download('students.pdf');
    }

    public function mark_alumni(Request $request)
    {
        $studentIds = $request->input('students', []);

        // Update the status of selected students to alumni
        Student::whereIn('id', $studentIds)->update(['status' => 'completed']);
        Student_classe::whereIn('student_id', $studentIds)->where('status', 'active')->update(['status' => 'completed']);
        Admission::whereIn('student_id', $studentIds)
            ->latest() // Sort by created_at or id
            ->groupBy('student_id')
            ->update(['status' => 'completed']);


        return redirect()->back()->with('success', 'Selected students have been marked as alumni.');
    }

    public function mark_current(Request $request)
    {
        $studentIds = $request->input('students', []);

        // Update the status of selected students to alumni
        Student::whereIn('id', $studentIds)->update(['status' => 'active']);
        Student_classe::whereIn('student_id', $studentIds)->where('status', 'completed')->update(['status' => 'active']);
        Admission::whereIn('student_id', $studentIds)
            ->latest() // Sort by created_at or id
            ->groupBy('student_id')
            ->update(['status' => 'active']);
        return redirect()->back()->with('success', 'Selected students have been marked as current.');
    }

    public function mark_alumni_single($id)
    {
        Student::where('id', $id)->update(['status' => 'completed']);
        Student_classe::where('student_id', $id)->where('status', 'active')->update(['status' => 'completed']);
        Admission::where('student_id', $id)
            ->latest() // Sort by created_at or id
            ->groupBy('student_id')
            ->update(['status' => 'completed']);
        return redirect()->back()->with('success', 'Selected students have been marked as alumni.');
    }

    public function mark_current_single($id)

    {
        Student::where('id', $id)->update(['status' => 'active']);
        Student_classe::where('student_id', $id)->where('status', 'completed')->update(['status' => 'active']);
        Admission::where('student_id', $id)
            ->latest() // Sort by created_at or id
            ->groupBy('student_id')
            ->update(['status' => 'active']);
        return redirect()->back()->with('success', 'Selected students have been marked as current.');
    }


    public function currentStudentGrid(Request $request)
    {
        $students = Student::with(['previousEducation', 'guardian', 'student_classes', 'classes', 'sections'])
            ->where('status', 'active');

        // Apply class filter if selected
        if ($request->filled('class_filter')) {
            $students->whereHas('classes', function ($query) use ($request) {
                $query->where('id', $request->class_filter);
            });
        }

        // Apply section filter if selected
        if ($request->filled('section_filter')) {
            $students->whereHas('sections', function ($query) use ($request) {
                $query->where('id', $request->section_filter);
            });
        }

        $students = $students->paginate(10);
        $classes  = DB::table('classes')->get();  // Fetch all available classes
        $sections = DB::table('sections')->get(); // Fetch all available sections

        return view('student.current-students-grid', compact('students', 'classes', 'sections'));
    }

    public function studentAlumni(Request $request)
    {
        $students = Student::with(['previousEducation', 'guardian', 'student_classes', 'classes', 'sections'])->where('status', 'completed');
        $schoolSettings = School_setting::all()->pluck('setting_value', 'setting_key');
        // Apply search filter
        if ($request->filled('search_query')) {
            $students->where(function ($query) use ($request) {
                $query->where('student_name', 'like', '%' . $request->search_query . '%')
                    ->orWhere('email', 'like', '%' . $request->search_query . '%');
            });
        }

        // Apply class filter if selected
        if ($request->filled('class_filter')) {
            $students->whereHas('latestAdmission', function ($query) use ($request) {
                $query->where('class_id', $request->class_filter);
            });
        }

        // Apply section filter if selected
        if ($request->filled('section_filter')) {
            $students->whereHas('latestAdmission', function ($query) use ($request) {
                $query->where('section_id', $request->section_filter);
            });
        }


        $students = $students->paginate(10);

        if ($request->ajax()) {
            return view('student.alumni-student-search-results', compact('students'))->render();
        }

        $classes  = DB::table('classes')->get();
        $sections = DB::table('sections')->get();

        return view('student.alumni-students', compact('students', 'classes', 'sections', 'schoolSettings'));
    }

    public function pdfAlumniStudent(Request $request)
    {
        $students = Student::with(['previousEducation', 'guardian', 'student_classes', 'classes', 'sections'])
            ->where('status', 'completed');

        // Apply search filter
        if ($request->filled('search_query')) {
            $students->where(function ($query) use ($request) {
                $query->where('student_name', 'like', '%' . $request->search_query . '%')
                    ->orWhere('email', 'like', '%' . $request->search_query . '%');
            });
        }

        // Apply class filter
        if ($request->filled('class_filter')) {
            $students->whereHas('latestAdmission', function ($query) use ($request) {
                $query->where('class_id', $request->class_filter);
            });
        }

        // Apply section filter
        if ($request->filled('section_filter')) {
            $students->whereHas('latestAdmission', function ($query) use ($request) {
                $query->where('section_id', $request->section_filter);
            });
        }

        // Get all matching students (no pagination)
        $students = $students->get();

        $pdf = Pdf::loadView('student.pdfAlumniStudent', compact('students'))->setPaper('A4', 'portrait');
        return $pdf->download('alumni_students.pdf');
    }


    //student profile
    public function profile($id)
    {

        $student = Student::with(['admissions', 'previousEducation', 'guardian', 'emergencyContact', 'documents', 'classes', 'sections'])
            ->where('id', $id)
            ->firstOrFail(); // Ensures a 404 error if no student is found


        return view('student.student-profile', compact('student'));
    }

    // student edit
    public function edit($id)
    {

        $student = Student::with(['admissions', 'previousEducation', 'guardian', 'emergencyContact', 'documents', 'classes', 'sections'])
            ->where('id', $id)
            ->firstOrFail(); // Ensures a 404 error if no student is found
        $classes = Classe::all();
        $sections = Section::all();
        $fee_structures = Fee_structure::all();
        return view('student.student-edit', compact('student', 'classes', 'sections', 'fee_structures'));
    }

    //update student
    public function update(Request $request, $id)
    {
        // dd($id);
        // Validate incoming request
        $request->validate([
            'student_name'      => 'required|string|max:255',
            'father_name'       => 'required|string|max:255',
            'father_cnic'       => 'required|string|max:15',
            'email'             => 'required|email|max:255',
            'religion'          => 'required|string|max:50',
            'mother_name'       => 'required|string|max:255',
            'gender'            => 'required|in:Male,Female,Other',
            'place_of_birth'    => 'required|string|max:255',
            'date_of_birth'     => 'required|date',
            'permanent_address' => 'required|string|max:500',
            'current_address'   => 'required|string|max:500',
            'blood_group'       => 'required|string|max:5',
        ]);

        // Find the student
        $student = Student::find($id);

        if (!$student) {
            return redirect()->back()->with('error', 'Student not found.');
        }

        // Update only changed fields
        $student->update([
            'student_name'      => $request->student_name,
            'father_name'       => $request->father_name,
            'father_cnic'       => $request->father_cnic,
            'email'             => $request->email,
            'religion'          => $request->religion,
            'mother_name'       => $request->mother_name,
            'gender'            => $request->gender,
            'place_of_birth'    => $request->place_of_birth,
            'date_of_birth'     => $request->date_of_birth,
            'permanent_address' => $request->permanent_address,
            'current_address'   => $request->current_address,
            'blood_group'       => $request->blood_group,
        ]);

        //update admission table for class and section
        if ($request->has('class') && $request->has('class_section')) {
            $student->latestAdmission()->update([
                'class_id'   => $request->class,
                'section_id' => $request->class_section,
            ]);
        }

        // Find the guardian by student_id
        $guardian = Guardian::where('student_id', $student->id)->first();

        if ($guardian) {
            // Update the existing guardian
            $guardian->update([
                'guardian_name'         => $request->guardian_name,
                'guardian_contact'      => $request->guardian_contact,
                'relation_with_student' => $request->guardian_relation,
            ]);
        }

        if ($request->has('previous_education')) {
            $edu = $request->previous_education;

            // Ensure the array has required keys to prevent errors
            if (!empty($edu['school_name']) && !empty($edu['last_class_passed']) && !empty($edu['class_to_apply']) && !empty($edu['passing_year'])) {
                // Find the previous education record by student_id
                $previousEducation = Previous_education::where('student_id', $id)->first();

                // Update the existing record
                $student->previousEducation()->updateOrCreate(
                    ['student_id' => $student->id], // Conditions to find
                    [                              // Attributes to update/create
                        'school_name'       => $edu['school_name'],
                        'last_class_passed' => $edu['last_class_passed'],
                        'class_to_apply'    => $edu['class_to_apply'],
                        'last_result'       => $edu['last_result'] ?? null,
                        'passing_year'      => $edu['passing_year']
                    ]
                );
            }
        }


        if ($request->filled(['emergency_contact_name', 'emergency_contact_number', 'relationship'])) {
            // Find the emergency contact record by student_id
            $emergencyContact = Emergency_contact::where('student_id', $student->id)->first();

            if ($emergencyContact) {
                // Update the existing record
                $emergencyContact->update([
                    'contact_name'   => $request->emergency_contact_name,
                    'contact_number' => $request->emergency_contact_number,
                    'relationship'   => $request->relationship,
                ]);
            }
        }
        return redirect()->back()->with('success', 'Student updated successfully!');
    }

    // student report screen
    public function reports_screens(Request $request)
    {
        // dd($request->all());
        $student = Student::with(['previousEducation', 'guardian', 'documents', 'student_classes', 'classes', 'sections'])
            ->whereHas('latestAdmission')
            ->with('latestAdmission')
            ->where('status', 'active');

        // Apply class filter if selected
        if ($request->filled('class')) {
            $student->whereHas('latestAdmission', function ($query) use ($request) {
                $query->where('class_id', $request->class);
            });
        }

        // Apply section filter if selected
        if ($request->filled('section')) {
            $student->whereHas('latestAdmission', function ($query) use ($request) {
                $query->where('section_id', $request->section);
            });
        }

        $students = $student->get();

        // $settings = school_setting::whereIn('setting_key', ['school_name', 'school_logo'])->pluck('setting_value', 'setting_key');
        $logo = school_setting::where('setting_key', 'school_logo')->first();
        $schoolName = school_setting::where('setting_key', 'school_name')->first();
        if ($logo) {
            $logo = url('school-logo/' . $logo->setting_value); // Assuming the logo is stored in the public disk
        } else {
            $logo = asset('assets/logo.png'); // Default value if not found
        }

        if ($schoolName) {
            $schoolName = $schoolName->setting_value;
        } else {
            $schoolName = 'School Name'; // Default value if not found
        }
        $cards = Student_cards_template::all();

        $classes = Classe::all();
        $sections = Section::all();
        return view('student.report-screens', compact('students', 'cards', 'classes', 'sections', 'logo', 'schoolName'));
    }

    public function stdAttendance($id, $month = 'current')
    {
        if ($month === 'current') {
            $month = now()->format('m'); // Get the current month
        }

        $attendances = DB::table('attendances')
            ->where('student_id', $id)
            ->whereMonth('date', $month)
            ->get();
        $student = Student::with(['previousEducation', 'guardian', 'documents', 'student_classes', 'classes', 'sections', 'attendances'])
            ->whereHas('latestAdmission')
            ->with('latestAdmission')
            ->where('status', 'active')
            ->where('id', $id)
            ->firstOrFail(); // Ensures a 404 error if no student is found
        return view('student.attendance-records', compact('id', 'student', 'attendances'));
    }
    public function stdDocuments($id)
    {
        $student = Student::with(['previousEducation', 'guardian', 'documents', 'student_classes', 'classes', 'sections'])
            ->whereHas('latestAdmission')
            ->with('latestAdmission')
            ->where('status', 'active')
            ->where('id', $id)
            ->firstOrFail(); // Ensures a 404 error if no student is found
        return view('student.documents-certificates', compact('id', 'student'));
    }
    public function stdAcademic($id)
    {
        $student = Student::with(['previousEducation', 'guardian', 'documents', 'student_classes', 'classes', 'sections'])
            ->whereHas('latestAdmission')
            ->with('latestAdmission')
            ->where('status', 'active')
            ->where('id', $id)
            ->firstOrFail(); // Ensures a 404 error if no student is found
        $subjects = Teacher_timetable::with('classes', 'sections', 'subjects', 'staff_member')
            ->where('class_id', $student->latestAdmission->class_id)
            ->where('section_id', $student->latestAdmission->section_id)
            ->get();
        return view('student.academic-information', compact('id', 'student', 'subjects'));
    }

    public function stdPromote($id)
    {
        $student = Student::with(['previousEducation', 'guardian', 'documents', 'student_classes', 'classes', 'sections'])
            ->whereHas('latestAdmission')
            ->with('latestAdmission')
            ->where('status', 'active')
            ->where('id', $id)
            ->firstOrFail(); // Ensures a 404 error if no student is found
        $classes = Classe::all();
        $sections = Section::all();
        return view('student.promote-student', compact('id', 'student', 'classes', 'sections'));
    }


    public function stdPromoteStore(Request $request)
    {
        $totalAmount = $request->input('total_amount');
        $currentClassId = $request->input('current_class_id');
        $nextClassId = $request->input('next_class_id');

        $fees = $request->input('fees', []);
        $feeDetails = [];
        $feeIds = [];

        foreach ($fees as $id => $fee) {
            // Only proceed if the checkbox was checked (i.e., 'amount' key is present)
            if (isset($fee['amount'])) {
                $amount = $fee['editable_amount'] ?? $fee['amount'];
                $type = $fee['type'] ?? "unknown_$id";

                $feeDetails[$type] = $amount;
                $feeIds[$type] = $id;
            }
        }

        DB::beginTransaction();

        try {
            $student = Student::where('status', 'active')
                ->where('id', $request->student_id)
                ->first();

            if (!$student) {
                return redirect()->back()->with('error', 'Student promotion failed. Student record not found or is not active.');
            }

            $currentAdmission = $student->latestAdmission()->firstOrFail();
            $latestStudentClass = $student->latestStudentClass()->firstOrFail();
            $latestAssignedFee = $student->latestAssignedFee()->firstOrFail();
            $sessionYear = School_session::where('status', 'active')->value('session_year');
            $student_id = $student->id;
            $head_code = $student->head_code;
            $reg_no = $student->reg_no;


            if (($nextClassId <= $currentClassId) || ($nextClassId > ($currentClassId + 1))) {
                return redirect()->back()->with('error', 'Please select Consective Class. Not jump higher ..!');
            }
            if (($nextClassId == $currentClassId) || ($currentAdmission->session_year == $sessionYear)) {
                return redirect()->back()->with('error', 'You do not promote to same class Or In the same session!');
            }


            $feeStructuresCheck = Fee_structure::where('class_id', $nextClassId)
                ->pluck('id', 'fee_type');

            if ($feeStructuresCheck->isEmpty()) {
                return redirect()->back()->with('error', 'Please First set the fee structure for the selected class!');
            }


            // Step 3: Update the Assigned Fee with latest fee IDs
            $latestAssignedFee->update([
                'class_id' => $nextClassId,
                'academic_year' => $sessionYear,
                'admission_fee' => null,
                'promotion_fee'      =>  null,
                'scholarship_fee'    =>   null,
                'sport_fee'          =>   null,
                'tuition_fee'        =>  $feeDetails['tuition_fee'] ?? null,
                'lab_fee'            =>  $feeDetails['lab_fee'] ?? null,
                'exam_fee'           =>  $feeDetails['exam_fee'] ?? null,
                'board_fee'          =>  $feeDetails['board_fee'] ?? null,
                'entertainment_fee'  =>  $feeDetails['entertainment_fee'] ?? null,
                'other_fee'          =>  $feeDetails['other_fee'] ?? null,
                'books_charges'      =>  $feeDetails['books_charges'] ?? null,
                'annual_charges'     =>  $feeDetails['annual_charges'] ?? null,
                'second_time_fee'    =>  $feeDetails['second_time_fee'] ?? null,
            ]);


            // Update Student Classe table
            $latestStudentClass->update([
                'academic_year' => $sessionYear,
                'class_id' => $nextClassId,
                'section_id' => $request->section_id ?? $currentAdmission->section_id,
            ]);

            $sessionYear = $currentAdmission->session_year; // e.g., "2025-2026"
            list($start, $end) = explode('-', $sessionYear);
            $previousSessionYear = ($start - 1) . '-' . ($end - 1);

            AdmissionHistory::create([
                'student_id' => $student_id,
                'head_code' => $head_code,
                'reg_no' => $reg_no,
                'admission_no' => $currentAdmission->admission_no,
                'class_id' => $currentAdmission->class_id,
                'section_id' => $currentAdmission->section_id,
                'session_year' => $previousSessionYear,
                'status' => 'completed',
                'admission_status' => 'promoted',
                'admission_date' => $currentAdmission->admission_date,
                'original_admission_id' => $currentAdmission->id,
                'promoted_at' => now(),
                'change_reason' => 'Class promotion'
            ]);

            // Update current admission table
            $currentAdmission->update([
                'session_year' => $sessionYear,
                'class_id' => $nextClassId,
                'section_id' => $request->section_id ?? $currentAdmission->section_id,
                'admission_status' => 'promoted',
                'admission_date' => now(),
            ]);

            $student_fee = Student_fee::create([
                'student_id'   => $student_id,
                'head_code'    => $head_code,
                'reg_no'       => $reg_no,
                'class_id'     => $nextClassId,
                'section_id'   => $request->section_id ?? $currentAdmission->section_id,
                'description'  => 'promotion',
                'status'       => 'unpaid',
                'discount'     => 0,
                'total_amount' => $totalAmount,
                'date'         => date('Y-m-d'),
            ]);

            Student_fee_detail::create([
                'student_id'         => $student_id,
                'head_code'          => $head_code,
                'reg_no'             => $reg_no,
                'class_id'           => $nextClassId,
                'std_fee_id'         => $student_fee->id,
                'academic_year'      => $sessionYear,
                'billing_month'      => null,
                'admission_fee'      => $feeDetails['admission_fee'] ?? null,
                'promotion_fee'      => $feeDetails['promotion_fee'] ?? null,
                'scholarship_fee'    => null,
                'sport_fee'          => null,
                'tuition_fee'        => $feeDetails['tuition_fee'] ?? null,
                'lab_fee'            => $feeDetails['lab_fee'] ?? null,
                'exam_fee'           => $feeDetails['exam_fee'] ?? null,
                'board_fee'          => $feeDetails['board_fee'] ?? null,
                'entertainment_fee'  => $feeDetails['entertainment_fee'] ?? null,
                'other_fee'          => $feeDetails['other_fee'] ?? null,
                'books_charges'      => $feeDetails['books_charges'] ?? null,
                'annual_charges'     => $feeDetails['annual_charges'] ?? null,
                'second_time_fee'    => $feeDetails['second_time_fee'] ?? null,
            ]);

            //Insert Transaction Data
            $transaction = Transaction::create([
                'transaction_type' => 'promotion',
                'trans_type_id' => $student_fee->id,
                'transaction_date' => date('Y-m-d'),
                'total_amount'     => $totalAmount,
                'remarks'          => 'promotion',
            ]);

            //Debit balance on student through head code
            Transaction_detail::create([
                'transaction_id'   => $transaction->id,
                'coa_id'           => $head_code,
                'transaction_type' => 'promotion',
                'credit'           => 0,
                'debit'            => ($totalAmount), // total_amount payble
            ]);

            //Credit balance from income through head code is 20400
            Transaction_detail::create([
                'transaction_id'   => $transaction->id,
                'coa_id'           => '20400',
                'transaction_type' => 'promotion',
                'credit'           => ($totalAmount),
                'debit'            => 0,
            ]);

            // Do work for remaining rupees of all students. we manage
            // accounts for it is receivable
            // head code for receivable is 20600
            Transaction_detail::create([
                'transaction_id'   => $transaction->id,
                'coa_id'           => '20600',
                'transaction_type' => 'promotion',
                'credit'           => 0,
                'debit'            => ($totalAmount),
            ]);

            DB::commit();
            return redirect()->back()->with('success', 'Student promoted successfully!');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()
                ->with('error', 'Promotion failed: ' . $e->getMessage());
        }
    }

    public function promoteClassForm()
    {
        $classes = Classe::all();
        return view('student.promte-bulk-students', compact('classes'));
    }

    public function promoteClass(Request $request)
    {

        $currentClassId = $request->input('current_class_id');
        $nextClassId = $request->input('next_class_id');
        $totalAmount = $request->input('total_amount');
        $sessionYear = School_session::where('status', 'active')->value('session_year');

        /*     if ($nextClassId == $currentClassId || $nextClassId < $currentClassId || $nextClassId == null || $currentClassId == null || ($nextClassId > $currentClassId + 1)) {
            return redirect()->back()->with('error', 'Please Correct Your Selection....');
        }
 */
        $fees = $request->input('fees', []);
        $feeDetails = [];
        $feeIds = [];

        foreach ($fees as $id => $fee) {
            // Only proceed if the checkbox was checked (i.e., 'amount' key is present)
            if (isset($fee['amount'])) {
                $amount = $fee['editable_amount'] ?? $fee['amount'];
                $type = $fee['type'] ?? "unknown_$id";

                $feeDetails[$type] = $amount;
                $feeIds[$type] = $id;
            }
        }

        DB::beginTransaction();

        try {

            $students = Admission::where('class_id', $currentClassId)
                ->where('status', 'active')
                ->get();
            if ($students->isEmpty()) {
                return redirect()->back()->with('error', 'No students found in the current class.');
            }
            foreach ($students as $student) {

                $sessionYear = $student->session_year; // e.g., "2025-2026"
                list($start, $end) = explode('-', $sessionYear);
                $previousSessionYear = ($start - 1) . '-' . ($end - 1);

                $checkPromotion = AdmissionHistory::where('student_id', $student->student_id)->where('class_id', $currentClassId)->where('session_year', $previousSessionYear)->first();
                if (!$checkPromotion) {
                    // Insert into AdmissionHistory
                    AdmissionHistory::create([
                        'student_id' => $student->student_id,
                        'head_code' => $student->head_code,
                        'reg_no' => $student->reg_no,
                        'admission_no' => $student->admission_no,
                        'class_id' => $currentClassId, // store OLD class_id here
                        'section_id' => $student->section_id,
                        'session_year' => $previousSessionYear,
                        'status' => 'completed',
                        'admission_status' => 'promoted',
                        'admission_date' => $student->admission_date,
                        'original_admission_id' => $student->id,
                        'promoted_at' => now(),
                        'change_reason' => 'Bulk class promotion'
                    ]);

                    // Update the current admission
                    $student->update([
                        'session_year' => $sessionYear,
                        'class_id' => $nextClassId,
                        'admission_status' => 'promoted'
                    ]);

                    $student_fee = Student_fee::create([
                        'student_id'   => $student->student_id,
                        'head_code'    => $student->head_code,
                        'reg_no'       => $student->reg_no,
                        'class_id'     => $nextClassId,
                        'section_id'   => $student->section_id,
                        'description'  => 'promotion',
                        'status'       => 'unpaid',
                        'discount'     => 0,
                        'total_amount' => $totalAmount,
                        'date'         => date('Y-m-d'),
                    ]);

                    Student_fee_detail::create([
                        'student_id'         => $student->student_id,
                        'head_code'          => $student->head_code,
                        'reg_no'             => $student->reg_no,
                        'class_id'           => $nextClassId,
                        'std_fee_id'         => $student_fee->id,
                        'academic_year'      => $sessionYear,
                        'billing_month'      => null,
                        'admission_fee'      => $feeDetails['admission_fee'] ?? null,
                        'promotion_fee'      => $feeDetails['promotion_fee'] ?? null,
                        'scholarship_fee'    => null,
                        'sport_fee'          => null,
                        'tuition_fee'        => $feeDetails['tuition_fee'] ?? null,
                        'lab_fee'            => $feeDetails['lab_fee'] ?? null,
                        'exam_fee'           => $feeDetails['exam_fee'] ?? null,
                        'board_fee'          => $feeDetails['board_fee'] ?? null,
                        'entertainment_fee'  => $feeDetails['entertainment_fee'] ?? null,
                        'other_fee'          => $feeDetails['other_fee'] ?? null,
                        'books_charges'      => $feeDetails['books_charges'] ?? null,
                        'annual_charges'     => $feeDetails['annual_charges'] ?? null,
                        'second_time_fee'    => $feeDetails['second_time_fee'] ?? null,
                    ]);



                    Student_assign_fee::updateOrCreate(
                        [
                            'student_id'   => $student->student_id,
                            'head_code' => $student->head_code,
                            'reg_no' => $student->reg_no,      // ← Search condition
                        ],
                        [
                            'class_id'     => $nextClassId,
                            'academic_year' => $sessionYear,
                            'admission_fee'        => null,
                            'promotion_fee'        => null,
                            'scholarship_fee'      => null,
                            'sport_fee'            => null,
                            'tuition_fee'          => $feeDetails['tuition_fee'] ?? null,
                            'lab_fee'              => $feeDetails['lab_fee'] ?? null,
                            'exam_fee'             => $feeDetails['exam_fee'] ?? null,
                            'board_fee'            => $feeDetails['board_fee'] ?? null,
                            'entertainment_fee'    => $feeDetails['entertainment_fee'] ?? null,
                            'other_fee'            => $feeDetails['other_fee'] ?? null,
                            'books_charges'        => $feeDetails['books_charges'] ?? null,
                            'annual_charges'       => $feeDetails['annual_charges'] ?? null,
                            'second_time_fee'      => $feeDetails['second_time_fee'] ?? null,
                        ]
                    );



                    Student_classe::where('student_id', $student->student_id)
                        ->where('status', 'active')
                        ->update([
                            'academic_year' => $sessionYear,
                            'class_id' => $nextClassId,
                            'status' => 'active'
                        ]);

                    //Insert Transaction Data
                    $transaction = Transaction::create([
                        'transaction_type' => 'promotion',
                        'trans_type_id' => $student_fee->id,
                        'transaction_date' => date('Y-m-d'),
                        'total_amount'     => $totalAmount,
                        'remarks'          => 'promotion',
                    ]);

                    //Debit balance on student through head code
                    Transaction_detail::create([
                        'transaction_id'   => $transaction->id,
                        'coa_id'           => $student->head_code,
                        'transaction_type' => 'promotion',
                        'credit'           => 0,
                        'debit'            => ($totalAmount), // total_amount payble
                    ]);

                    //Credit balance from income through head code is 20400
                    Transaction_detail::create([
                        'transaction_id'   => $transaction->id,
                        'coa_id'           => '20400',
                        'transaction_type' => 'promotion',
                        'credit'           => ($totalAmount),
                        'debit'            => 0,
                    ]);

                    // Do work for remaining rupees of all students. we manage
                    // accounts for it is receivable
                    // head code for receivable is 20600
                    Transaction_detail::create([
                        'transaction_id'   => $transaction->id,
                        'coa_id'           => '20600',
                        'transaction_type' => 'promotion',
                        'credit'           => 0,
                        'debit'            => ($totalAmount),
                    ]);
                }
            }


            DB::commit();
            return redirect()->back()->with('success', "Total  {$students->count()}  Student promoted successfully to class!");
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Promotion failed: ' . $e->getMessage());
        }
    }
}
