<?php

namespace App\Http\Controllers;

use App\Models\Job_position;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Barryvdh\DomPDF\Facade\Pdf;

class JobPositionController extends Controller
{
    //All Jobs
    public function index(Request $request)
    {
        $job_openings = Job_position::query()
            ->orderBy('created_at', 'desc');

        if ($request->filled('search_query')) {
            $job_openings->where(function ($query) use ($request) {
                $query->where('title', 'like', '%' . $request->search_query . '%')
                    ->orWhere('category', 'like', '%' . $request->search_query . '%')
                    ->orWhere('qualification_required', 'like', '%' . $request->search_query . '%')
                    ->orWhere('description', 'like', '%' . $request->search_query . '%')
                    ->orWhere('salary', 'like', '%' . $request->search_query . '%')
                    ->orWhere('status', 'like', '%' . $request->search_query . '%');
            });
        }
        if ($request->filled('category')) {
            $job_openings->where(function ($query) use ($request) {
                    $query->Where('category', 'like', '%' . $request->category . '%');
            });
        }

        $job_openings = $job_openings->paginate(10);

        if ($request->ajax()) {
            return view('jobPosition.partialPosition', compact('job_openings'))->render();
        }

        return view('jobPosition.allPositions', compact('job_openings'));
    }

    //Open jobs
    public function indexOpen(Request $request)
    {
        $job_openings = Job_position::where('status', 'open')
            ->orderBy('created_at', 'desc');

        if ($request->filled('search_query')) {
            $job_openings->where(function ($query) use ($request) {
                $query->where('title', 'like', '%' . $request->search_query . '%')
                    ->orWhere('category', 'like', '%' . $request->search_query . '%')
                    ->orWhere('qualification_required', 'like', '%' . $request->search_query . '%')
                    ->orWhere('description', 'like', '%' . $request->search_query . '%')
                    ->orWhere('salary', 'like', '%' . $request->search_query . '%')
                    ->orWhere('status', 'like', '%' . $request->search_query . '%');
            });
        }
        if ($request->filled('category')) {
            $job_openings->where(function ($query) use ($request) {
                $query->Where('category', 'like', '%' . $request->category . '%');
            });
        }

        $job_openings = $job_openings->paginate(10);

        if ($request->ajax()) {
            return view('jobPosition.partialOpenPosition', compact('job_openings'))->render();
        }

        return view('jobPosition.openPositon', compact('job_openings'));
    }

    //Close jobs
    public function indexClose(Request $request)
    {
        $job_openings = Job_position::where('status', 'closed')
            ->orderBy('created_at', 'desc');

        if ($request->filled('search_query')) {
            $job_openings->where(function ($query) use ($request) {
                $query->where('title', 'like', '%' . $request->search_query . '%')
                    ->orWhere('category', 'like', '%' . $request->search_query . '%')
                    ->orWhere('qualification_required', 'like', '%' . $request->search_query . '%')
                    ->orWhere('description', 'like', '%' . $request->search_query . '%')
                    ->orWhere('salary', 'like', '%' . $request->search_query . '%')
                    ->orWhere('status', 'like', '%' . $request->search_query . '%');
            });
        }
        if ($request->filled('category')) {
            $job_openings->where(function ($query) use ($request) {
                $query->Where('category', 'like', '%' . $request->category . '%');
            });
        }

        $job_openings = $job_openings->paginate(10);

        if ($request->ajax()) {
            return view('jobPosition.partialClosePosition', compact('job_openings'))->render();
        }

        return view('jobPosition.closePosition', compact('job_openings'));
    }

    public function create()
    {

        return view('jobPosition.positionForm');
    }

    public function store(Request $request)
    {
        // Step 1: Validate inputs (without 'unique')
        $request->validate([
            'title'                  => 'required|string|max:255',
            'category'               => 'required|string|max:255',
            'age'                    => 'required|integer|min:18|max:65',
            'qualification_required' => 'required|string',
            'experience_required'    => 'required|integer|min:0|max:50',
            'salary'                 => 'required|numeric|min:0',
            'description'            => 'required|string|min:10',
        ]);

        // Step 2: Custom check for duplicate category or description
        $duplicate = job_position::where('category', $request->category)
            ->orWhere('description', $request->description)
            ->first();

        if ($duplicate) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'A job with this department/subject or description already exists.');
        }

        // Step 3: Proceed to insert if no duplicate found
        DB::beginTransaction();

        try {
            job_position::create([
                'title'                => $request->title,
                'category'             => $request->category,
                'qualification_required' => $request->qualification_required,
                'experience_required'  => $request->experience_required,
                'age'                  => $request->age,
                'salary'               => $request->salary,
                'description'          => $request->description
            ]);

            DB::commit();
            return redirect()->route('job-positions')
                ->with('success', 'Job Position added successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('job-positions-create')
                ->withInput()
                ->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    public function edit(Request $request)
    {
        $validatedData = $request->validate([
            'id' => 'required|integer',
            'title' => 'required|string|max:255',
            'category' => 'required|string|max:255',
            'qualification_required' => 'required|string',
            'experience_required' => 'required|string',
            'description' => 'nullable|string',
            'salary' => 'required|numeric|min:0',
            'status' => 'required',
        ]);

        DB::beginTransaction();

        try {
            $jobPosition = Job_position::findOrFail($validatedData['id']);

            // Custom duplicate check: excluding current record
            $duplicate = Job_position::where(function ($query) use ($validatedData) {
                $query->where('category', $validatedData['category'])
                    ->orWhere('description', $validatedData['description']);
            })
                ->where('id', '!=', $validatedData['id'])
                ->first();

            if ($duplicate) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'A job with this department/subject or description already exists.');
            }

            // Update
            $jobPosition->update([
                'title' => $validatedData['title'],
                'category' => $validatedData['category'],
                'qualification_required' => $validatedData['qualification_required'],
                'experience_required' => $validatedData['experience_required'],
                'description' => $validatedData['description'] ?? null,
                'salary' => $validatedData['salary'] ?? 0,
                'status' => $validatedData['status'],
            ]);

            DB::commit();

            return redirect()->back()->with('success', 'Job position updated successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Job position update failed: ' . $e->getMessage());

            return redirect()->back()
                ->with('error', 'Failed to update job position. Please try again.')
                ->withInput();
        }
    }

    public function delete($id)
    {
        $job = Job_position::findOrFail($id);
        $job->delete();
        return redirect()->back()->with('success', 'Job position updated successfully!');
    }


    public function pdfAllJobsPosition(Request $request)
    {
        $allJob = Job_position::with('job_applications')->orderBy('created_at', 'desc')->get();

        $pdf = Pdf::loadView('jobPosition.pdfAllJobPosition', compact('allJob'));

        return $pdf->download('all_job_positions.pdf');
    }

    public function pdfOpenJobsPosition(Request $request)
    {
        $openJob = Job_position::with('job_applications')->where('status','open')->orderBy('created_at', 'desc')->get();

        $pdf = Pdf::loadView('jobPosition.pdfOpenJobPosition', compact('openJob'));

        return $pdf->download('open_job_positions.pdf');
    }

    public function pdfCloseJobsPosition(Request $request)
    {
        $closeJob = Job_position::with('job_applications')->where('status', 'closed')->orderBy('created_at', 'desc')->get();

        $pdf = Pdf::loadView('jobPosition.pdfCloseJobPosition', compact('closeJob'));

        return $pdf->download('close_job_positions.pdf');
    }
}
